/* $Id: measwatchdog.c,v 1.2 1999/03/04 23:29:00 ericb Exp $ */
/* Copyright (C) 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Template for E1432 test programs */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include "e1432.h"

#define	BLOCKSIZE	1024
#define	NSCAN		32
#define	NMOD_MAX	4
#define	NCHAN_MAX	(NMOD_MAX * E1432_INPUT_CHANS)

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: measwatchdog.c,v 1.2 1999/03/04 23:29:00 ericb Exp $";
static const char *progname;

static int
monitor(E1432ID hw, const char *str, int verbose, int should_be_small)
{
    LONGSIZ32 watchdog;

    CHECK(e1432_read32_register(hw, E1432_INPUT_CHAN(1),
				E1432_MEAS_WATCHDOG_REG, &watchdog));
    if (verbose)
	(void) printf("%s: watchdog %ld\n", str, watchdog);

    if (should_be_small)
    {
	if (watchdog > 1)
	{
	    (void) fprintf(stderr,
			   "%s: watchdog register %ld, expected 0 or 1\n",
			   progname, watchdog);
	    return -1;
	}
    }
    else
    {
	if (watchdog < 2)
	{
	    (void) fprintf(stderr,
			   "%s: watchdog register %ld, expected > 1\n",
			   progname, watchdog);
	    return -1;
	}
    }
	

    return 0;
}

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw, int *group,
     int *nchan, SHORTSIZ16 *chan_list)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, nc;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    for (i = 0; i < nmod; i++)
	nc += hwconfig[i].input_chans;
    if (nc > NCHAN_MAX)
	nc = NCHAN_MAX;
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    *nchan = nc;

    for (i = 0; i < nc; i++)
	chan_list[i] = E1432_INPUT_CHAN(i + 1);

    *group = e1432_create_channel_group(*hw, nc, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    return 0;
}

/*ARGSUSED*/
static int
setup(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
      int freq, int magsq, int verbose)
{
    CHECK(e1432_reset_measure(hw, group));
    CHECK(monitor(hw, "Setup1", verbose, 1));
    CHECK(e1432_set_append_status(hw, group, E1432_APPEND_STATUS_ON));
    CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));
    if (freq)
    {
	CHECK(e1432_set_calc_data(hw, group, E1432_DATA_FREQ));
	if (magsq)
	{
	    CHECK(e1432_set_avg_mode(hw, group, E1432_AVG_RMS));
	    CHECK(e1432_set_avg_number(hw, group, 1));
	}
    }
    CHECK(monitor(hw, "Setup2", verbose, 1));

    return 0;
}

static int
check_trailer(struct e1432_trailer *trailer, FLOATSIZ32 clock_freq,
	      double span, int chan, int type)
{
    double  tmp;
    int     df2, df5;

    if (trailer->zoom_corr != 0)
    {
	/* Zoom correction is not currently implemented */
	(void) fprintf(stderr,
		       "%s: trailer zoom corr non-zero: %g (0x%lx)\n",
		       progname, trailer->zoom_corr,
		       *(long *) &trailer->zoom_corr);
	return -1;
    }
    if (trailer->gap < 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer gap negative: 0x%lx\n",
		       progname, trailer->gap);
	return -1;
    }
    if (trailer->rpm1 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm1 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm1,
		       *(long *) &trailer->rpm1);
	return -1;
    }
    if (trailer->rpm2 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm2 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm2,
		       *(long *) &trailer->rpm2);
	return -1;
    }
    if (trailer->peak != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer peak non-zero: %g (0x%lx)\n",
		       progname, trailer->peak,
		       *(long *) &trailer->peak);
	return -1;
    }
    if (trailer->rms != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rms non-zero: %g (0x%lx)\n",
		       progname, trailer->rms,
		       *(long *) &trailer->rms);
	return -1;
    }

    /* Compute df2 and df5 from clock_freq and span */
    tmp = span * 2.56;
    df2 = 0;
    df5 = 0;
    while (tmp < clock_freq * 0.9999)
    {
	df2++;
	tmp *= 2;
    }
    if (tmp > clock_freq * 1.0001)
    {
	tmp /= 8;
	tmp *= 5;
	df2 -= 3;
	df5++;
	if (tmp > clock_freq * 1.0001 || tmp < clock_freq * 0.9999)
	{
	    (void) fprintf(stderr,
			   "%s: invalid span/clock_freq combination: %g/%g\n",
			   progname, span, clock_freq);
	    return -1;
	}
    }

    if (df2 != ((trailer->info & E1432_TRAILER_INFO_DEC_2_MASK)
		>> E1432_TRAILER_INFO_DEC_2_SHIFT))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df2 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df2);
	return -1;
    }
    if (df5 != ((trailer->info & E1432_TRAILER_INFO_DEC_5) != 0))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df5 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df5);
	return -1;
    }

    if (((trailer->info & E1432_TRAILER_INFO_CHAN_MASK) >>
	 E1432_TRAILER_INFO_CHAN_SHIFT) != chan - 1)
    {
	(void) fprintf(stderr,
		       "%s: trailer info chan mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, chan - 1);
	return -1;
    }
    if (((trailer->info & E1432_TRAILER_INFO_TYPE_MASK) >>
	 E1432_TRAILER_INFO_TYPE_SHIFT) != type)
    {
	(void) fprintf(stderr,
		       "%s: trailer info type mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, type);
	return -1;
    }

    return 0;
}

static int
wait_block_avail(E1432ID hw, int group, int scan, int verbose,
		 long blocksize, double span)
{
    clock_t start, timeout;
    int     status;

    timeout = (2 + 2 * (blocksize / (span * 2.56))) * CLOCKS_PER_SEC;
    if (verbose > 2)
	(void) printf("Waiting %g sec for block available\n",
		      (double) timeout / CLOCKS_PER_SEC);
    start = clock();
    while ((status = e1432_block_available(hw, group)) == 0)
	if (clock() - start > timeout &&
	    (status = e1432_block_available(hw, group)) == 0)
	{
	    (void) fprintf(stderr, "%s: e1432_block_available: "
			   "timeout waiting %g sec\n",
			   progname, (double) timeout / CLOCKS_PER_SEC);
	    return -1;
	}
    if (status < 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_block_available: returned %d\n",
		       progname, status);
	return -1;
    }
    if (verbose > 0)
	(void) printf("Scan %d block available found\n", scan);

    return 0;
}

static int
run(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
    int freq, int magsq, int verbose)
{
    FLOATSIZ64 buffer[BLOCKSIZE];
    LONGSIZ32 count, expect;
    struct e1432_trailer trailer;
    int     scan, type, type_max, chan;

    CHECK(monitor(hw, "Pre init meas", verbose, 1));
    CHECK(e1432_init_measure(hw, group));
    CHECK(monitor(hw, "Post init meas", verbose, 0));

    for (scan = 0; scan < NSCAN; scan++)
    {
	/* Wait for block available */
	CHECK(monitor(hw, "Pre block avail", verbose, 0));
	CHECK(wait_block_avail(hw, group, scan, verbose,
			       BLOCKSIZE, 20000));
	CHECK(monitor(hw, "Post block avail", verbose, 0));

	type_max = freq ? 1 : 0;

	/* Read the data */
	for (type = 0; type <= type_max; type++)
	    for (chan = 0; chan < nchan; chan++)
	    {
		if (verbose > 1)
		    (void) printf("Reading chan %d type %d\n",
				  chan, type);

		expect = BLOCKSIZE;
		if (type != 0 && magsq)
		    expect = BLOCKSIZE / 2;
		CHECK(e1432_read_float64_data(hw, chan_list[chan],
					      type == 0 ?
					      E1432_TIME_DATA :
					      E1432_FREQ_DATA,
					      buffer, expect,
					      &trailer,
					      &count));
		if (count != expect)
		{
		    (void) fprintf(stderr,
				   "%s: e1432_read_float64_data: "
				   "actual count was %ld\n",
				   progname, count);
		    return -1;
		}

		CHECK(check_trailer(&trailer, 51200, 20000,
				    chan_list[chan], type));
	    }
    }

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-fmuvV] [-L laddr] [-n nchan] [-N nmod]\n"
		   "\t-f: Set up for freq data\n"
		   "\t-L: First logical address is <laddr>, default 8\n"
		   "\t-m: Send mag-squared data, not complex, for freq data\n"
		   "\t-n: Use <nchan> channels, default -1 meaning all\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    SHORTSIZ16 chan_list[NCHAN_MAX];
    char   *p;
    int     c, i, freq, magsq, nmod, verbose;
    int     group, nchan;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    freq = 0;
    laddr[0] = 8;
    magsq = 0;
    nchan = -1;			/* Meaning use all channels */
    nmod = 1;
    verbose = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "fL:mn:N:uvV")) != -1)
	switch (c)
	{
	case 'f':
	    freq = 1;
	    break;
	case 'L':
	    laddr[0] = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'm':
	    magsq = 1;
	    break;
	case 'n':
	    nchan = strtol(optarg, &p, 0);
	    if (optarg == p || nchan < -1 || nchan > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    /* Run the measurement */
    if (init(nmod, laddr, &hw, &group, &nchan, chan_list) < 0)
	return EXIT_FAILURE;
    if (setup(hw, group, nchan, chan_list, freq, magsq, verbose) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list, freq, magsq, verbose) < 0)
	return EXIT_FAILURE;
    if (setup(hw, group, nchan, chan_list, freq, magsq, verbose) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list, freq, magsq, verbose) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
